local assets =
{
    Asset("ANIM", "anim/nightstick.zip"),
    Asset("ANIM", "anim/swap_nightstick.zip"),
    Asset("SOUND", "sound/common.fsb"),
}

local prefabs =
{
    "nightstickfire",
    "electrichitsparks",
}

local function onpocket(inst)
    inst.components.burnable:Extinguish()
end

local function onremovefire(fire)
    fire.nightstick.fire = nil
end

local function onequip(inst, owner)
    inst.components.burnable:Ignite()

    local skin_build = inst:GetSkinBuild()
    if skin_build ~= nil then
        owner:PushEvent("equipskinneditem", inst:GetSkinName())
        owner.AnimState:OverrideItemSkinSymbol("swap_object", skin_build, "swap_nightstick", inst.GUID, "swap_nightstick")
    else
        owner.AnimState:OverrideSymbol("swap_object", "swap_nightstick", "swap_nightstick")
    end

    owner.AnimState:Show("ARM_carry")
    owner.AnimState:Hide("ARM_normal")

    inst.SoundEmitter:PlaySound("dontstarve_DLC001/common/morningstar", "torch")
    --inst.SoundEmitter:SetParameter("torch", "intensity", 1)

    if inst.fire == nil then
        inst.fire = SpawnPrefab("nightstickfire")
        inst.fire.nightstick = inst
        inst:ListenForEvent("onremove", onremovefire, inst.fire)
    end
    inst.fire.entity:SetParent(owner.entity)
end

local function onunequip(inst, owner)
    local skin_build = inst:GetSkinBuild()
    if skin_build ~= nil then
        owner:PushEvent("unequipskinneditem", inst:GetSkinName())
    end

    if inst.fire ~= nil then
        inst.fire:Remove()
    end

    inst.components.burnable:Extinguish()
    owner.AnimState:Hide("ARM_carry")
    owner.AnimState:Show("ARM_normal")
    inst.SoundEmitter:KillSound("torch")
end

local function onequiptomodel(inst, owner, from_ground)
    if inst.fire ~= nil then
        inst.fire:Remove()
    end

    inst.components.burnable:Extinguish()
    inst.SoundEmitter:KillSound("torch")
end

local function OnRemoveEntity(inst)
    if inst.fire ~= nil then
        inst.fire:Remove()
    end
end

local function onfuelchange(newsection, oldsection, inst)
    if newsection <= 0 then
        --when we burn out
        if inst.components.burnable ~= nil then
            inst.components.burnable:Extinguish()
        end
        local equippable = inst.components.equippable
        if equippable ~= nil and equippable:IsEquipped() then
            local owner = inst.components.inventoryitem ~= nil and inst.components.inventoryitem.owner or nil
            if owner ~= nil then
                local data =
                {
                    prefab = inst.prefab,
                    equipslot = equippable.equipslot,
                    announce = "ANNOUNCE_TORCH_OUT",
                }
                inst:Remove()
                owner:PushEvent("itemranout", data)
                return
            end
        end
        inst:Remove()
    end
end

local function onattack(inst, attacker, target)
    SpawnElectricHitSparks(attacker, target, true)
end

local function OnBatteryUsed(inst, battery)
    if inst.components.fueled:IsFull() then
        return false, "CHARGE_FULL"
    end

    local newpercent = math.clamp(inst.components.fueled:GetPercent() + TUNING.NIGHTSTICK_BATTERYCHARGE, 0, 1)
    inst.components.fueled:SetPercent(newpercent)
    SpawnElectricHitSparks(inst, battery, true)

    return true
end

local function fn()
    local inst = CreateEntity()

    inst.entity:AddTransform()
    inst.entity:AddAnimState()
    inst.entity:AddSoundEmitter()
    inst.entity:AddNetwork()

    inst.AnimState:SetBank("nightstick")
    inst.AnimState:SetBuild("nightstick")
    inst.AnimState:PlayAnimation("idle")

    MakeInventoryPhysics(inst)

    inst:AddTag("wildfireprotected")

    --batteryuser (from batteryuser component) added to pristine state for optimization
    --inst:AddTag("batteryuser") -- FIXME(JBK): WF: Uncomment this up when strings.

    --moonsparkchargeable (from moonsparkchargeable component) added to pristine state for optimization
    inst:AddTag("moonsparkchargeable")

    --weapon (from weapon component) added to pristine state for optimization
    inst:AddTag("weapon")

    MakeInventoryFloatable(inst, "med", 0.05, {1.1, 0.4, 1.1}, true, -19, {sym_build = "swap_nightstick"})

    inst.entity:SetPristine()

    if not TheWorld.ismastersim then
        return inst
    end

    inst:AddComponent("weapon")
    inst.components.weapon:SetDamage(TUNING.NIGHTSTICK_DAMAGE)
    inst.components.weapon:SetOnAttack(onattack)
    inst.components.weapon:SetElectric()

    inst:AddComponent("inventoryitem")
    -----------------------------------

    inst:AddComponent("equippable")
    inst.components.equippable:SetOnPocket(onpocket)
    inst.components.equippable:SetOnEquip(onequip)
    inst.components.equippable:SetOnUnequip(onunequip)
    inst.components.equippable:SetOnEquipToModel(onequiptomodel)

    -----------------------------------

    inst:AddComponent("inspectable")

    -----------------------------------

    inst:AddComponent("burnable")
    inst.components.burnable.canlight = false
    inst.components.burnable.fxprefab = nil

    -----------------------------------

    inst:AddComponent("fueled")
    inst.components.fueled:SetSectionCallback(onfuelchange)
    inst.components.fueled:InitializeFuelLevel(TUNING.NIGHTSTICK_FUEL)
    inst.components.fueled:SetDepletedFn(inst.Remove)
    inst.components.fueled:SetFirstPeriod(TUNING.TURNON_FUELED_CONSUMPTION, TUNING.TURNON_FULL_FUELED_CONSUMPTION)

    inst:AddComponent("moonsparkchargeable")
    inst.components.moonsparkchargeable:SetFueledPercent(TUNING.NIGHTSTICK_SPARKCHARGE)

    --inst:AddComponent("batteryuser") -- FIXME(JBK): WF: Uncomment this up when strings.
    --inst.components.batteryuser.onbatteryused = OnBatteryUsed -- FIXME(JBK): WF: Uncomment this up when strings.

    MakeHauntableLaunch(inst)

    inst.OnRemoveEntity = OnRemoveEntity

    return inst
end

return Prefab("nightstick", fn, assets, prefabs)
